"use strict";
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const he = __importStar(require("he"));
const dom_1 = require("./utils/dom");
const types_1 = require("./utils/types");
const xml_1 = require("./utils/xml");
const xpath_1 = require("./utils/xpath");
// tslint:disable:prefer-for-of
// The main entry point of the XSL-T processor, as explained above.
//
// @param input The input document root, as XPath ExprContext.
// @param template The stylesheet document root, as DOM node.
// @param the root of the generated output, as DOM node.
function xsltProcessContext(input, template, templateRoot, output) {
    const outputDocument = xml_1.xmlOwnerDocument(output);
    const nodename = template.nodeName.split(/:/);
    if (nodename.length === 1 || nodename[0] !== 'xsl') {
        xsltPassThrough(input, template, templateRoot, output, outputDocument);
    }
    else {
        // let name: string;
        // let top: Element;
        // let nameexpr: string;
        // let node: Node;
        // let select: string;
        // let value: XPathResult;
        // let nodes: Node[];
        // let sortContext;
        // let mode: string;
        // let templates: Element[];
        // let paramContext;
        // let commentData: string;
        // let commentNode: Comment;
        // let test: string;
        // let match: string;
        // let text: string;
        switch (nodename[1]) {
            case 'apply-imports': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'apply-templates': {
                const select = xmlGetAttribute(template, 'select');
                let nodes;
                if (select) {
                    nodes = xpath_1.gatherNodes(input.eval(select));
                }
                else {
                    nodes = Array.from(input.node.childNodes);
                }
                const sortContext = input.clone(nodes[0], input.rootNode, 0, nodes);
                xsltWithParam(sortContext, template, templateRoot);
                xsltSort(sortContext, template);
                const mode = xmlGetAttribute(template, 'mode');
                const top = templateRoot;
                const templates = [];
                for (let i = 0; i < top.childNodes.length; ++i) {
                    const c = top.childNodes[i];
                    if (types_1.isElement(c) && c.nodeName === 'xsl:template' && (!mode || c.getAttribute('mode') === mode)) {
                        templates.push(c);
                    }
                }
                for (let j = 0; j < sortContext.contextSize(); ++j) {
                    const nj = sortContext.nodeList[j];
                    for (let i = 0; i < templates.length; ++i) {
                        xsltProcessContext(sortContext.clone(nj, sortContext.rootNode, j), templates[i], templateRoot, output);
                    }
                }
                break;
            }
            case 'attribute': {
                const nameexpr = xmlGetAttribute(template, 'name');
                const name = xsltAttributeValue(nameexpr, input);
                const node = dom_1.domCreateDocumentFragment(outputDocument);
                xsltChildNodes(input, template, templateRoot, node);
                const v = xml_1.xmlValue(node);
                if (types_1.isElement(output)) {
                    dom_1.domSetAttribute(output, name, v);
                }
                else {
                    throw new Error('Can not set attribute on non-element node.');
                }
                break;
            }
            case 'attribute-set': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'call-template': {
                const name = xmlGetAttribute(template, 'name');
                const top = templateRoot;
                const paramContext = input.clone();
                xsltWithParam(paramContext, template, templateRoot);
                for (let i = 0; i < top.childNodes.length; ++i) {
                    const c = top.childNodes[i];
                    if (types_1.isElement(c) && c.nodeName === 'xsl:template' && dom_1.domGetAttribute(c, 'name') === name) {
                        xsltChildNodes(paramContext, c, templateRoot, output);
                        break;
                    }
                }
                break;
            }
            case 'choose': {
                xsltChoose(input, template, templateRoot, output);
                break;
            }
            case 'comment': {
                const node = dom_1.domCreateDocumentFragment(outputDocument);
                xsltChildNodes(input, template, templateRoot, node);
                const commentData = xml_1.xmlValue(node);
                const commentNode = dom_1.domCreateComment(outputDocument, commentData);
                output.appendChild(commentNode);
                break;
            }
            case 'copy': {
                const node = xsltCopy(output, input.node, outputDocument);
                if (node !== undefined) {
                    xsltChildNodes(input, template, templateRoot, node);
                }
                break;
            }
            case 'copy-of': {
                const select = xmlGetAttribute(template, 'select');
                const value = input.eval(select);
                if (xpath_1.isResultNodeSet(value)) {
                    const nodes = xpath_1.gatherNodes(value);
                    for (let i = 0; i < nodes.length; ++i) {
                        xsltCopyOf(output, nodes[i], outputDocument);
                    }
                }
                else {
                    const node = dom_1.domCreateTextNode(outputDocument, value.stringValue);
                    dom_1.domAppendChild(output, node);
                }
                break;
            }
            case 'decimal-format': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'element': {
                const nameexpr = xmlGetAttribute(template, 'name');
                const name = xsltAttributeValue(nameexpr, input);
                const node = dom_1.domCreateElement(outputDocument, name);
                dom_1.domAppendChild(output, node);
                xsltChildNodes(input, template, templateRoot, node);
                break;
            }
            case 'fallback': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'for-each': {
                xsltForEach(input, template, templateRoot, output);
                break;
            }
            case 'if': {
                const test = xmlGetAttribute(template, 'test');
                if (input.eval(test).booleanValue) {
                    xsltChildNodes(input, template, templateRoot, output);
                }
                break;
            }
            case 'import': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'include': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'key': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'message': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'namespace-alias': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'number': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'otherwise': {
                throw new Error(`error if here: ${nodename[1]}`);
            }
            case 'output': {
                // Ignored. -- Since we operate on the DOM, and all further use
                // of the output of the XSL transformation is determined by the
                // browser that we run in, this parameter is not applicable to
                // this implementation.
                break;
            }
            case 'preserve-space': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'processing-instruction': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'sort': {
                // just ignore -- was handled by xsltSort()
                break;
            }
            case 'strip-space': {
                throw new Error(`not implemented: ${nodename[1]}`);
            }
            case 'stylesheet':
            case 'transform': {
                xsltChildNodes(input, template, templateRoot, output);
                break;
            }
            case 'template': {
                const match = xmlGetAttribute(template, 'match');
                if (match && xsltMatch(match, input)) {
                    xsltChildNodes(input, template, templateRoot, output);
                }
                break;
            }
            case 'text': {
                const text = xml_1.xmlValue(template);
                const node = dom_1.domCreateTextNode(outputDocument, text);
                output.appendChild(node);
                break;
            }
            case 'value-of': {
                const select = xmlGetAttribute(template, 'select');
                const value = input.eval(select);
                const node = dom_1.domCreateTextNode(outputDocument, value.stringValue);
                output.appendChild(node);
                break;
            }
            case 'param': {
                xsltVariable(input, template, templateRoot, false);
                break;
            }
            case 'variable': {
                xsltVariable(input, template, templateRoot, true);
                break;
            }
            case 'when': {
                throw new Error(`error if here: ${nodename[1]}`);
            }
            case 'with-param': {
                throw new Error(`error if here: ${nodename[1]}`);
            }
            default: {
                throw new Error(`error if here: ${nodename[1]}`);
            }
        }
    }
}
exports.xsltProcessContext = xsltProcessContext;
// Sets parameters defined by xsl:with-param child nodes of the
// current template node, in the current input context. This happens
// before the operation specified by the current template node is
// executed.
function xsltWithParam(input, template, templateRoot) {
    for (const c of Array.from(template.childNodes)) {
        if (types_1.isElement(c) && c.nodeName === 'xsl:with-param') {
            xsltVariable(input, c, templateRoot, true);
        }
    }
}
// Orders the current node list in the input context according to the
// sort order specified by xsl:sort child nodes of the current
// template node. This happens before the operation specified by the
// current template node is executed.
//
// TODO(mesch): case-order is not implemented.
function xsltSort(input, template) {
    const sort = [];
    for (const c of Array.from(template.childNodes)) {
        if (types_1.isElement(c) && c.nodeName === 'xsl:sort') {
            const select = xmlGetAttribute(c, 'select');
            const type = xmlGetAttribute(c, 'data-type') || 'text';
            const order = xmlGetAttribute(c, 'order') || 'ascending';
            sort.push({
                expr: select,
                type,
                order
            });
        }
    }
    xpath_1.xpathSort(input, sort);
}
// Evaluates a variable or parameter and set it in the current input
// context. Implements xsl:variable, xsl:param, and xsl:with-param.
//
// @param override flag that defines if the value computed here
// overrides the one already in the input context if that is the
// case. I.e. decides if this is a default value or a local
// value. xsl:variable and xsl:with-param override; xsl:param doesn't.
function xsltVariable(input, template, templateRoot, override) {
    const name = xmlGetAttribute(template, 'name');
    const select = xmlGetAttribute(template, 'select');
    let value;
    if (template.childNodes.length > 0) {
        const root = dom_1.domCreateDocumentFragment(template.ownerDocument);
        xsltChildNodes(input, template, templateRoot, root);
        value = [root];
    }
    else if (select) {
        value = xpath_1.convertResult(input.eval(select));
    }
    else {
        value = '';
    }
    if (override || !input.getVariable(name)) {
        input.setVariable(name, value);
    }
}
// Implements xsl:chose and its child nodes xsl:when and
// xsl:otherwise.
function xsltChoose(input, template, templateRoot, output) {
    for (const childNode of Array.from(template.childNodes)) {
        if (!types_1.isElement(childNode)) {
            continue;
        }
        else if (childNode.nodeName === 'xsl:when') {
            const test = xmlGetAttribute(childNode, 'test');
            if (input.eval(test).booleanValue) {
                xsltChildNodes(input, childNode, templateRoot, output);
                break;
            }
        }
        else if (childNode.nodeName === 'xsl:otherwise') {
            xsltChildNodes(input, childNode, templateRoot, output);
            break;
        }
    }
}
// Implements xsl:for-each.
function xsltForEach(input, template, templateRoot, output) {
    const select = xmlGetAttribute(template, 'select');
    const nodes = xpath_1.gatherNodes(input.eval(select));
    const sortContext = input.clone(nodes[0], input.rootNode, 0, nodes);
    xsltSort(sortContext, template);
    for (let i = 0; i < sortContext.contextSize(); ++i) {
        const ni = sortContext.nodeList[i];
        xsltChildNodes(sortContext.clone(ni, input.rootNode, i), template, templateRoot, output);
    }
}
// Traverses the template node tree. Calls the main processing
// function with the current input context for every child node of the
// current template node.
function xsltChildNodes(input, template, templateRoot, output) {
    // Clone input context to keep variables declared here local to the
    // siblings of the children.
    const context = input.clone();
    for (let i = 0; i < template.childNodes.length; ++i) {
        if (isXmlDeclaration(template.childNodes[i]) || types_1.isComment(template.childNodes[i])) {
            continue;
        }
        xsltProcessContext(context, template.childNodes[i], templateRoot, output);
    }
}
// Passes template text to the output. The current template node does
// not specify an XSL-T operation and therefore is appended to the
// output with all its attributes. Then continues traversing the
// template node tree.
function xsltPassThrough(input, template, templateRoot, output, outputDocument) {
    if (types_1.isText(template)) {
        if (xsltPassText(template)) {
            const node = dom_1.domCreateTextNode(outputDocument, template.nodeValue);
            dom_1.domAppendChild(output, node);
        }
    }
    else if (types_1.isElement(template)) {
        const node = dom_1.domCreateElement(outputDocument, template.nodeName);
        for (const a of Array.from(template.attributes)) {
            if (a) {
                const name = a.nodeName;
                const value = xsltAttributeValue(a.nodeValue, input);
                dom_1.domSetAttribute(node, name, value);
            }
        }
        dom_1.domAppendChild(output, node);
        xsltChildNodes(input, template, templateRoot, node);
    }
    else {
        // This applies also to the DOCUMENT_NODE of the XSL stylesheet,
        // so we don't have to treat it specially.
        xsltChildNodes(input, template, templateRoot, output);
    }
}
// Determines if a text node in the XSLT template document is to be
// stripped according to XSLT whitespace stipping rules.
//
// See [XSLT], section 3.4.
//
// TODO(mesch): Whitespace stripping on the input document is
// currently not implemented.
function xsltPassText(template) {
    if (!template.nodeValue.match(/^\s*$/)) {
        return true;
    }
    let element = template.parentNode;
    if (element.nodeName === 'xsl:text') {
        return true;
    }
    while (element && types_1.isElement(element)) {
        const xmlspace = dom_1.domGetAttribute(element, 'xml:space');
        if (xmlspace) {
            if (xmlspace === 'default') {
                return false;
            }
            else if (xmlspace === 'preserve') {
                return true;
            }
        }
        element = element.parentNode;
    }
    return false;
}
// Evaluates an XSL-T attribute value template. Attribute value
// templates are attributes on XSL-T elements that contain XPath
// expressions in braces {}. The XSL-T expressions are evaluated in
// the current input context.
function xsltAttributeValue(value, context) {
    const parts = value.split('{');
    if (parts.length === 1) {
        return value;
    }
    let ret = '';
    for (let i = 0; i < parts.length; ++i) {
        const rp = parts[i].split('}');
        if (rp.length !== 2) {
            // first literal part of the value
            ret += parts[i];
            continue;
        }
        const val = context.eval(rp[0]).stringValue;
        ret += val + rp[1];
    }
    return ret;
}
// Wrapper function to access attribute values of template element
// nodes. Currently this calls he.decode because in some DOM
// implementations the return value of node.getAttributeValue()
// contains unresolved XML entities, although the DOM spec requires
// that entity references are resolved by te DOM.
function xmlGetAttribute(node, name) {
    // TODO(mesch): This should not be necessary if the DOM is working
    // correctly. The DOM is responsible for resolving entities, not the
    // application.
    if (types_1.isElement(node)) {
        const value = dom_1.domGetAttribute(node, name);
        if (value != null) {
            return he.decode(value);
        }
        else {
            return '';
        }
    }
    else {
        throw new Error('Can not get attribute from non-Element node');
    }
}
/**
 * Implements xsl:copy-of for node-set values of the select
 * expression. Recurses down the source node tree, which is part of
 * the input document.
 * @param dst the node being copied to, part of output document,
 * @param src the node being copied, part in input document,
 * @param dstDocument
 */
function xsltCopyOf(dst, src, dstDocument) {
    if (types_1.isFragment(src) || types_1.isDocument(src)) {
        for (let i = 0; i < src.childNodes.length; ++i) {
            xsltCopyOf(dst, src.childNodes[i], dstDocument);
        }
    }
    else if (types_1.isElement(src)) {
        const node = xsltCopy(dst, src, dstDocument);
        if (node !== undefined) {
            // This was an element node -- recurse to attributes and
            // children.
            for (let i = 0; i < src.attributes.length; ++i) {
                xsltCopyOf(node, src.attributes[i], dstDocument);
            }
            for (let i = 0; i < src.childNodes.length; ++i) {
                xsltCopyOf(node, src.childNodes[i], dstDocument);
            }
        }
    }
}
/**
 * Implements xsl:copy for all node types.
 * @param dst the node being copied to, part of output document,
 * @param src the node being copied, part in input document,
 * @param dstDocument
 * @return If an element node was created, the element
 * node. Otherwise undefined.
 */
function xsltCopy(dst, src, dstDocument) {
    if (types_1.isElement(src)) {
        const node = dom_1.domCreateElement(dstDocument, src.nodeName);
        dom_1.domAppendChild(dst, node);
        return node;
    }
    if (types_1.isText(src)) {
        const node = dom_1.domCreateTextNode(dstDocument, src.nodeValue);
        dom_1.domAppendChild(dst, node);
    }
    else if (types_1.isCData(src)) {
        const node = dom_1.domCreateCDATASection(dstDocument, src.nodeValue);
        dom_1.domAppendChild(dst, node);
    }
    else if (types_1.isComment(src)) {
        const node = dom_1.domCreateComment(dstDocument, src.nodeValue);
        dom_1.domAppendChild(dst, node);
    }
    else if (types_1.isAttribute(src)) {
        if (types_1.isElement(dst)) {
            dom_1.domSetAttribute(dst, src.nodeName, src.nodeValue);
        }
        else {
            throw new Error('Can not set attribute to non-element node.');
        }
    }
    return undefined;
}
// Evaluates an XPath expression in the current input context as a
// match (see [XSLT] section 5.2, paragraph 1).
function xsltMatch(match, context) {
    let ret;
    ret = false;
    let node = context.node;
    while (!ret && node) {
        const cloned = context.clone(node, context.rootNode, 0, [node]);
        const result = xpath_1.gatherNodes(cloned.eval(match));
        for (let i = 0; i < result.length; ++i) {
            if (result[i] === context.node) {
                ret = true;
                break;
            }
        }
        node = node.parentNode;
    }
    return ret;
}
function isXmlDeclaration(node) {
    if (types_1.isProcessingInstruction(node)) {
        return node.target.startsWith('xml');
    }
    else {
        return false;
    }
}
//# sourceMappingURL=xslt.js.map