"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const types_1 = require("./types");
// tslint:disable:prefer-for-of
// Returns the text value of a node; for nodes without children this
// is the nodeValue, for nodes with children this is the concatenation
// of the value of all children. Browser-specific optimizations are used by
// default; they can be disabled by passing "true" in as the second parameter.
function xmlValue(node, disallowBrowserSpecificOptimization = false) {
    if (!node) {
        return '';
    }
    let ret = '';
    if (types_1.isText(node) || types_1.isCData(node)) {
        ret += node.nodeValue;
    }
    else if (types_1.isAttribute(node)) {
        ret += node.nodeValue;
    }
    else if (types_1.isElement(node) || types_1.isDocument(node) || types_1.isFragment(node)) {
        if (!disallowBrowserSpecificOptimization) {
            // IE, Safari, Opera, and friends
            try {
                const innerText = node.innerText;
                if (innerText != null) {
                    return innerText;
                }
            }
            catch (_e) {
                // ignore if implemented but not supported
            }
            // Firefox
            const textContent = node.textContent;
            if (textContent != null) {
                return textContent || '';
            }
        }
        // pobrecito!
        const len = node.childNodes.length;
        for (let i = 0; i < len; ++i) {
            ret += xmlValue(node.childNodes[i]);
        }
    }
    return ret;
}
exports.xmlValue = xmlValue;
// Returns the representation of a node as XML text.
function xmlText(node, optCdata = false) {
    const buf = [];
    xmlTextR(node, buf, optCdata);
    return buf.join('');
}
exports.xmlText = xmlText;
function xmlTextR(node, buf, cdata) {
    if (types_1.isText(node)) {
        buf.push(xmlEscapeText(node.nodeValue));
    }
    else if (types_1.isCData(node)) {
        if (cdata) {
            buf.push(node.nodeValue);
        }
        else {
            buf.push(`<![CDATA[${node.nodeValue}]]>`);
        }
    }
    else if (types_1.isComment(node)) {
        buf.push(`<!--${node.nodeValue}-->`);
    }
    else if (types_1.isElement(node)) {
        buf.push(`<${xmlFullNodeName(node)}`);
        for (let i = 0; i < node.attributes.length; ++i) {
            const a = node.attributes[i];
            if (a && a.nodeName && a.nodeValue) {
                buf.push(` ${xmlFullNodeName(a)}="${xmlEscapeAttr(a.nodeValue)}"`);
            }
        }
        if (node.childNodes.length === 0) {
            buf.push('/>');
        }
        else {
            buf.push('>');
            for (let i = 0; i < node.childNodes.length; ++i) {
                xmlTextR(node.childNodes[i], buf, cdata);
            }
            buf.push(`</${xmlFullNodeName(node)}>`);
        }
    }
    else if (types_1.isDocument(node) || types_1.isFragment(node)) {
        for (let i = 0; i < node.childNodes.length; ++i) {
            xmlTextR(node.childNodes[i], buf, cdata);
        }
    }
}
function xmlFullNodeName(n) {
    if (n.prefix && n.nodeName.indexOf(`${n.prefix}:`) !== 0) {
        return `${n.prefix}:${n.nodeName}`;
    }
    else {
        return n.nodeName;
    }
}
// Escape XML special markup chracters: tag delimiter < > and entity
// reference start delimiter &. The escaped string can be used in XML
// text portions (i.e. between tags).
function xmlEscapeText(s) {
    return `${s}`
        .replace(/&/g, '&amp;')
        .replace(/&amp;amp;/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;');
}
exports.xmlEscapeText = xmlEscapeText;
// Escape XML special markup characters: tag delimiter < > entity
// reference start delimiter & and quotes ". The escaped string can be
// used in double quoted XML attribute value portions (i.e. in
// attributes within start tags).
function xmlEscapeAttr(s) {
    return xmlEscapeText(s).replace(/"/g, '&quot;');
}
/**
 * Wrapper function to access the owner document uniformly for document
 * and other nodes: for the document node, the owner document is the
 * node itself, for all others it's the ownerDocument property.
 *
 */
function xmlOwnerDocument(node) {
    if (types_1.isDocument(node)) {
        return node;
    }
    else {
        return node.ownerDocument;
    }
}
exports.xmlOwnerDocument = xmlOwnerDocument;
//# sourceMappingURL=xml.js.map