"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const function_resolver_1 = require("./function-resolver");
const namespace_resolver_1 = require("./namespace-resolver");
const variable_resolver_1 = require("./variable-resolver");
const xpath_parser_1 = require("./xpath-parser");
const xpath_types_1 = require("./xpath-types");
const parser = new xpath_parser_1.XPathParser();
const defaultNSResolver = new namespace_resolver_1.NamespaceResolverImpl();
const defaultFunctionResolver = new function_resolver_1.FunctionResolverImpl();
const defaultVariableResolver = new variable_resolver_1.VariableResolverImpl();
function makeNSResolverFromFunction(func) {
    return {
        getNamespace(prefix, node) {
            const ns = func(prefix, node);
            return ns || defaultNSResolver.getNamespace(prefix, node);
        }
    };
}
function makeNSResolverFromObject(obj) {
    return makeNSResolverFromFunction(obj.getNamespace.bind(obj));
}
function makeNSResolverFromMap(map) {
    return makeNSResolverFromFunction((prefix) => {
        return map[prefix];
    });
}
function makeNSResolver(resolver) {
    if (resolver && typeof resolver.getNamespace === 'function') {
        return makeNSResolverFromObject(resolver);
    }
    if (typeof resolver === 'function') {
        return makeNSResolverFromFunction(resolver);
    }
    // assume prefix -> uri mapping
    if (typeof resolver === 'object') {
        return makeNSResolverFromMap(resolver);
    }
    return defaultNSResolver;
}
exports.makeNSResolver = makeNSResolver;
/** Converts native JavaScript types to their XPath library equivalent */
function convertValue(value) {
    if (value == null) {
        return null;
    }
    if (value instanceof xpath_types_1.XString || value instanceof xpath_types_1.XBoolean || value instanceof xpath_types_1.XNumber || value instanceof xpath_types_1.XNodeSet) {
        return value;
    }
    switch (typeof value) {
        case 'string':
            return new xpath_types_1.XString(value);
        case 'boolean':
            return new xpath_types_1.XBoolean(value);
        case 'number':
            return new xpath_types_1.XNumber(value);
    }
    // assume node(s)
    const ns = new xpath_types_1.XNodeSet();
    ns.addArray([].concat(value));
    return ns;
}
exports.convertValue = convertValue;
function makeEvaluator(func) {
    return (context, ...args) => {
        args = args.map((a) => a.evaluate(context));
        const result = func(context, ...args);
        return convertValue(result); // if result is not null convertValue will not return null
    };
}
function makeFunctionResolverFromFunction(func) {
    return {
        getFunction(name, namespace) {
            const found = func(name, namespace);
            if (found != null) {
                return makeEvaluator(found);
            }
            return defaultFunctionResolver.getFunction(name, namespace);
        }
    };
}
function makeFunctionResolverFromObject(obj) {
    return makeFunctionResolverFromFunction(obj.getFunction.bind(obj));
}
function makeFunctionResolverFromMap(map) {
    return makeFunctionResolverFromFunction((name) => {
        return map[name];
    });
}
function makeFunctionResolver(resolver) {
    if (resolver && typeof resolver.getFunction === 'function') {
        return makeFunctionResolverFromObject(resolver);
    }
    if (typeof resolver === 'function') {
        return makeFunctionResolverFromFunction(resolver);
    }
    // assume map
    if (typeof resolver === 'object') {
        return makeFunctionResolverFromMap(resolver);
    }
    return defaultFunctionResolver;
}
exports.makeFunctionResolver = makeFunctionResolver;
function makeVariableResolverFromFunction(func) {
    return {
        getVariable(name, namespace) {
            const value = func(name, namespace);
            return convertValue(value);
        }
    };
}
function makeVariableResolver(resolver) {
    if (resolver) {
        if (typeof resolver.getVariable === 'function') {
            return makeVariableResolverFromFunction(resolver.getVariable.bind(resolver));
        }
        if (typeof resolver === 'function') {
            return makeVariableResolverFromFunction(resolver);
        }
        // assume map
        if (typeof resolver === 'object') {
            return makeVariableResolverFromFunction((name) => {
                return resolver[name];
            });
        }
    }
    return defaultVariableResolver;
}
exports.makeVariableResolver = makeVariableResolver;
function copyIfPresent(prop, dest, source) {
    if (prop in source) {
        dest[prop] = source[prop];
    }
}
function makeContext(options) {
    const context = new xpath_types_1.XPathContext(new variable_resolver_1.VariableResolverImpl(), new namespace_resolver_1.NamespaceResolverImpl(), new function_resolver_1.FunctionResolverImpl());
    if (options !== undefined) {
        context.namespaceResolver = makeNSResolver(options.namespaces);
        context.functionResolver = makeFunctionResolver(options.functions);
        context.variableResolver = makeVariableResolver(options.variables);
        context.expressionContextNode = options.node;
        copyIfPresent('allowAnyNamespaceForNoPrefix', context, options);
        copyIfPresent('isHtml', context, options);
    }
    else {
        context.namespaceResolver = defaultNSResolver;
    }
    return context;
}
function evaluate(parsedExpression, options) {
    const context = makeContext(options);
    return parsedExpression.evaluate(context);
}
function parse(xpath) {
    const parsed = parser.parse(xpath);
    return new class {
        constructor() {
            this.expression = parsed;
        }
        evaluate(options) {
            return evaluate(this.expression, options);
        }
        evaluateNumber(options) {
            return this.evaluate(options).numberValue;
        }
        evaluateString(options) {
            return this.evaluate(options).stringValue;
        }
        evaluateBoolean(options) {
            return this.evaluate(options).booleanValue;
        }
        evaluateNodeSet(options) {
            return this.evaluate(options).nodeset;
        }
        select(options) {
            return this.evaluateNodeSet(options).toArray();
        }
        select1(options) {
            return this.select(options)[0];
        }
    }();
}
exports.parse = parse;
//# sourceMappingURL=parse-api.js.map