"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const dom_implementation_1 = require("../dom-implementation");
class DOMHandlerImpl {
    constructor() {
        this.cdata = false;
        this.currentElement = null;
    }
    startDocument() {
        this.doc = new dom_implementation_1.DOMImplementationImpl().createDocument(null, null, null);
        if (this.locator) {
            this.doc.documentURI = this.locator.systemId || '';
        }
    }
    startElement(namespaceURI, localName, qName, attrs) {
        const doc = this.doc;
        const el = doc.createElementNS(namespaceURI, qName || localName);
        const len = attrs.length;
        this.appendElement(el);
        this.currentElement = el;
        if (this.locator) {
            position(this.locator, el);
        }
        for (let i = 0; i < len; i++) {
            const attrNamespaceURI = attrs.getURI(i);
            const value = attrs.getValue(i);
            const attrQName = attrs.getQName(i);
            const attr = doc.createAttributeNS(attrNamespaceURI || null, attrQName);
            if (this.locator) {
                position(attrs.getLocator(i) || null, attr);
            }
            attr.value = attr.nodeValue = value;
            el.setAttributeNode(attr);
        }
    }
    endElement(_namespaceURI, _localName, _qName) {
        const current = this.currentElement;
        if (current === null) {
            throw new Error('No open element');
        }
        // const tagName = current.tagName;
        this.currentElement = current.parentNode;
    }
    startPrefixMapping(_prefix, _uri) {
        // empty
    }
    endPrefixMapping(_prefix) {
        // empty
    }
    processingInstruction(target, data) {
        const ins = this.doc.createProcessingInstruction(target, data);
        if (this.locator) {
            position(this.locator, ins);
        }
        this.appendElement(ins);
    }
    ignorableWhitespace(_ch, _start, _length) {
        // empty
    }
    characters(chars, start, length) {
        chars = chars.substr(start, length);
        // console.log(chars)
        if (chars) {
            let charNode;
            if (this.cdata) {
                charNode = this.doc.createCDATASection(chars);
            }
            else {
                charNode = this.doc.createTextNode(chars);
            }
            if (this.currentElement) {
                this.currentElement.appendChild(charNode);
            }
            else if (/^\s*$/.test(chars)) {
                this.doc.appendChild(charNode);
                // process xml
            }
            if (this.locator) {
                position(this.locator, charNode);
            }
        }
    }
    skippedEntity(_name) {
        // empty
    }
    endDocument() {
        this.doc.normalize();
    }
    setDocumentLocator(locator) {
        this.locator = locator;
        if (this.locator) {
            // && !('lineNumber' in locator)){
            locator.lineNumber = 0;
        }
    }
    // LexicalHandler
    comment(chars, start, length) {
        chars = chars.substr(start, length);
        const comm = this.doc.createComment(chars);
        if (this.locator) {
            position(this.locator, comm);
        }
        this.appendElement(comm);
    }
    startCDATA() {
        // used in characters() methods
        this.cdata = true;
    }
    endCDATA() {
        this.cdata = false;
    }
    startDTD(name, publicId, systemId) {
        const impl = this.doc.implementation;
        if (impl && impl.createDocumentType) {
            const dt = impl.createDocumentType(name, publicId, systemId);
            if (this.locator) {
                position(this.locator, dt);
            }
            this.appendElement(dt);
        }
    }
    endDTD() {
        // empty
    }
    warning(error) {
        // tslint:disable-next-line: no-console
        console.warn('[xmldom warning]\t' + error, _locator(this.locator));
    }
    error(error) {
        // tslint:disable-next-line: no-console
        console.error('[xmldom error]\t' + error, _locator(this.locator));
    }
    fatalError(error) {
        // tslint:disable-next-line: no-console
        console.error('[xmldom fatalError]\t' + error, _locator(this.locator));
    }
    appendElement(node) {
        if (!this.currentElement) {
            this.doc.appendChild(node);
        }
        else {
            this.currentElement.appendChild(node);
        }
    } // appendChild and setAttributeNS are preformance key
}
exports.DOMHandlerImpl = DOMHandlerImpl;
function position(locator, node) {
    if (locator) {
        node.lineNumber = locator.lineNumber;
        node.columnNumber = locator.columnNumber;
    }
}
/**
 * @see org.xml.sax.ContentHandler#startDocument
 * @link http://www.saxproject.org/apidoc/org/xml/sax/ContentHandler.html
 */
function _locator(l) {
    if (l) {
        return '\n@' + (l.systemId || '') + '#[line:' + l.lineNumber + ',col:' + l.columnNumber + ']';
    }
}
//# sourceMappingURL=dom-handler.js.map