"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("./types");
const document_utils_1 = require("./document-utils");
const dom_exception_1 = require("./dom-exception");
const dummy_element_1 = require("./dummy/dummy-element");
const live_node_list_1 = require("./live-node-list");
const node_list_1 = require("./node-list");
const node_types_1 = require("./node-types");
const utils_1 = require("./utils");
class ElementImpl extends dummy_element_1.DummyElement {
    constructor() {
        super();
        this._nsMap = {};
        this.nodeType = node_types_1.NodeTypeTS.ELEMENT_NODE;
    }
    hasAttribute(name) {
        return this.getAttributeNode(name) != null;
    }
    getAttribute(name) {
        const attr = this.getAttributeNode(name);
        return attr != null ? attr.value || '' : null;
    }
    getAttributeNode(name) {
        return this.attributes.getNamedItem(name);
    }
    setAttribute(name, value) {
        const attr = this.ownerDocument.createAttribute(name);
        attr.value = attr.nodeValue = '' + value;
        this.setAttributeNode(attr);
    }
    removeAttribute(name) {
        const attr = this.getAttributeNode(name);
        if (attr) {
            this.removeAttributeNode(attr);
        }
    }
    // four real opeartion method
    appendChild(newChild) {
        if (utils_1.isDocumentFragment(newChild)) {
            return this.insertBefore(newChild, null);
        }
        else {
            const _newChild = document_utils_1._appendSingleChild(this, utils_1.asChildNode(newChild));
            // notify observers
            this.queueMutation({
                type: 'childList',
                target: this,
                addedNodes: new node_list_1.NodeListImpl(_newChild),
                removedNodes: new node_list_1.NodeListImpl(),
                previousSibling: _newChild.previousSibling,
                nextSibling: _newChild.nextSibling,
                attributeName: null,
                attributeNamespace: null,
                oldValue: null,
            });
            return _newChild;
        }
    }
    setAttributeNode(newAttr) {
        if (this.ownerDocumentInternal() !== newAttr.ownerDocument) {
            throw new dom_exception_1.DOMExceptionImpl(dom_exception_1.DOMExceptionImpl.WRONG_DOCUMENT_ERR);
        }
        const oldAttr = this.attributes.setNamedItem(newAttr);
        // notify observers
        this.queueMutation({
            type: 'attributes',
            target: this,
            addedNodes: new node_list_1.NodeListImpl(),
            removedNodes: new node_list_1.NodeListImpl(),
            previousSibling: null,
            nextSibling: null,
            attributeName: newAttr.nodeName,
            attributeNamespace: null,
            oldValue: oldAttr != null ? oldAttr.value : null,
        });
        return oldAttr;
    }
    setAttributeNodeNS(newAttr) {
        if (this.ownerDocumentInternal() !== newAttr.ownerDocument) {
            throw new dom_exception_1.DOMExceptionImpl(dom_exception_1.DOMExceptionImpl.WRONG_DOCUMENT_ERR);
        }
        const oldAttr = this.attributes.setNamedItemNS(newAttr);
        // notify observers
        this.queueMutation({
            type: 'attributes',
            target: this,
            addedNodes: new node_list_1.NodeListImpl(),
            removedNodes: new node_list_1.NodeListImpl(),
            previousSibling: null,
            nextSibling: null,
            attributeName: newAttr.localName,
            attributeNamespace: newAttr.namespaceURI,
            oldValue: oldAttr != null ? oldAttr.value : null,
        });
        return oldAttr;
    }
    removeAttributeNode(attr) {
        // console.log(this == oldAttr.ownerElement)
        const oldAttr = this.attributes.removeNamedItem(attr.nodeName);
        // notify observers
        this.queueMutation({
            type: 'attributes',
            target: this,
            addedNodes: new node_list_1.NodeListImpl(),
            removedNodes: new node_list_1.NodeListImpl(),
            previousSibling: null,
            nextSibling: null,
            attributeName: oldAttr.namespaceURI != null ? oldAttr.localName : oldAttr.nodeName,
            attributeNamespace: oldAttr.namespaceURI,
            oldValue: oldAttr.value,
        });
        return oldAttr;
    }
    // get real attribute name,and remove it by removeAttributeNode
    removeAttributeNS(namespaceURI, localName) {
        const old = this.getAttributeNodeNS(namespaceURI, localName);
        if (old) {
            this.removeAttributeNode(old);
        }
    }
    hasAttributeNS(namespaceURI, localName) {
        return this.getAttributeNodeNS(namespaceURI, localName) != null;
    }
    getAttributeNS(namespaceURI, localName) {
        const attr = this.getAttributeNodeNS(namespaceURI, localName);
        return (attr && attr.value) || '';
    }
    setAttributeNS(namespaceURI, qualifiedName, value) {
        const attr = this.ownerDocument.createAttributeNS(namespaceURI, qualifiedName);
        attr.value = attr.nodeValue = '' + value;
        this.setAttributeNode(attr);
    }
    getAttributeNodeNS(namespaceURI, localName) {
        return this.attributes.getNamedItemNS(namespaceURI, localName);
    }
    getElementsByTagName(tagName) {
        return new live_node_list_1.LiveNodeListImpl(this, (base) => {
            const ls = [];
            document_utils_1._visitNode(base, (node) => {
                if (node !== base && utils_1.isElement(node) && (tagName === '*' || node.tagName === tagName)) {
                    ls.push(node);
                }
            });
            return ls;
        });
    }
    getElementsByTagNameNS(namespaceURI, localName) {
        return new live_node_list_1.LiveNodeListImpl(this, (base) => {
            const ls = [];
            document_utils_1._visitNode(base, (node) => {
                if (node !== base &&
                    utils_1.isElement(node) &&
                    (namespaceURI === '*' || node.namespaceURI === namespaceURI) &&
                    (localName === '*' || node.localName === localName)) {
                    ls.push(node);
                }
            });
            return ls;
        });
    }
    get outerHTML() {
        return this.toString();
    }
}
exports.ElementImpl = ElementImpl;
//# sourceMappingURL=element.js.map