/// <reference types="node" />
import { BaseEncodingOptions, ReadStream } from 'fs';
import { Visibility } from '../enum';
import { IFilesystemAdapter, IReadFileOptions, IMimeTypeDetector, IVisibilityConverter, IStorageAttributes, IPathStats, IFilesystemVisibility, RequireOne } from '../interfaces';
import { PathPrefixer, FileAttributes } from '../libs';
import { OPTION_DIRECTORY_VISIBILITY } from '../constant';
import { Readable } from 'stream';
/**
 * local file write config
 */
export declare type TLocalFilesystemAdapterWriteConfig = IFilesystemVisibility & BaseEncodingOptions & {
    flag?: string;
};
/**
 * local filesystem adapter
 */
export declare class LocalFilesystemAdapter implements IFilesystemAdapter {
    protected readonly _visibility: IVisibilityConverter;
    protected writeFlags: string;
    private linkHandling;
    protected mimeTypeDetector: IMimeTypeDetector;
    /**
     * 0001
     * @var number
     */
    static SKIP_LINKS: number;
    /**
     * 0002
     * @var number
     */
    static DISALLOW_LINKS: number;
    /**
     * @var string
     */
    protected pathSeparator: string;
    /**
     * @protected PathPrefixer
     */
    protected prefixer: PathPrefixer;
    /**
     * Constructor.
     *
     * @param {string} root
     * @param {IVisibilityConverter} _visibility
     * @param {number} writeFlags
     * @param {number} linkHandling
     * @param {IMimeTypeDetector} mimeTypeDetector
     *
     * @throws LogicException
     */
    constructor(root: string, _visibility?: IVisibilityConverter, writeFlags?: string, linkHandling?: number, mimeTypeDetector?: IMimeTypeDetector);
    /**
     * get path prefix
     */
    getPathPrefix(): PathPrefixer;
    fileExists(path: string): Promise<boolean>;
    /**
     * Ensure the root directory exists.
     *
     * @param {string} root root directory path
     * @param {visibility} visibility control
     * @return void
     *
     * @throws Exception in case the root directory can not be created
     */
    protected ensureDirectorySync(root: string, visibility: number): void;
    /**
     * Ensure the root directory exists.
     *
     * @param {string} root root directory path
     * @param {number} visibility path mode
     * @return void
     *
     * @throws Exception in case the root directory can not be created
     */
    protected ensureDirectoryExists(root: string, visibility?: number): Promise<void>;
    /**
     * resolve directory visibility
     * @param visibility
     * @protected
     */
    protected resolveDirectoryVisibility(visibility?: Visibility): number;
    /**
     * @inheritdoc
     */
    write(path: string, contents: string | Buffer, config?: TLocalFilesystemAdapterWriteConfig): Promise<void>;
    /**
     * @inheritdoc
     */
    writeStream(path: string, resource: Readable, config?: TLocalFilesystemAdapterWriteConfig): Promise<void>;
    /**
     * @inheritdoc
     */
    readStream(path: string, config?: any): Promise<ReadStream>;
    /**
     * @inheritdoc
     */
    read(path: string, config?: IReadFileOptions): Promise<Buffer | string>;
    /**
     * @inheritdoc
     */
    move(path: string, newPath: string): Promise<void>;
    /**
     * @inheritdoc
     */
    copy(path: string, newPath: string, config?: {
        [OPTION_DIRECTORY_VISIBILITY]?: Visibility;
    }): Promise<void>;
    /**
     * @inheritdoc
     */
    delete(path: string): Promise<void>;
    /**
     * @inheritdoc
     */
    listContents(directory?: string, recursive?: boolean): Promise<IStorageAttributes[]>;
    /**
     * @inheritdoc
     */
    fileSize(path: string): Promise<RequireOne<FileAttributes, 'fileSize'>>;
    /**
     * @inheritdoc
     * @param path
     */
    lastModified(path: string): Promise<RequireOne<FileAttributes, 'lastModified'>>;
    /**
     * @inheritdoc
     */
    mimeType(path: string): Promise<RequireOne<FileAttributes, 'mimeType'>>;
    /**
     * @inheritdoc
     */
    visibility(path: string): Promise<RequireOne<FileAttributes, 'visibility'>>;
    /**
     * @inheritdoc
     */
    setVisibility(path: string, visibility: Visibility | string): Promise<void>;
    /**
     * @inheritdoc
     */
    createDirectory(dirname: string, config?: IFilesystemVisibility): Promise<void>;
    /**
     * @inheritdoc
     */
    deleteDirectory(dirname: string): Promise<undefined>;
    /**
     * @param {IPathStats} file
     */
    /**
     * Normalize the file info.
     *
     * @param {Dirent} file
     *
     * @return array|void
     *
     * @throws NotSupportedException
     */
    protected normalizeFileInfo(file: IPathStats): IStorageAttributes | undefined;
    /**
     * Get the normalized path from a SplFileInfo object.
     *
     * @param {IPathStats} file
     *
     * @return string
     */
    protected getFilePath(file: IPathStats): string;
    /**
     * @param {IPathStats} file
     *
     * @return array
     */
    protected mapFileInfo(file: IPathStats): IStorageAttributes;
    /**
     * @param {IPathStats} file
     *
     * @throws UnreadableFileException
     */
    protected guardAgainstUnreadableFileInfo(file: IPathStats): Promise<void>;
    private setPermissions;
}
